// $Id: HNSTypedTensor.m,v 1.3 2003/03/20 13:39:57 hns Exp $
//
//
//  BasicData.m
//  CocoaBasic
//
//  Created by Dr. H. Nikolaus Schaller on Sat Jun 15 2002.
//  Copyright (c) 2001 __MyCompanyName__. All rights reserved.
//

#define INDEX_ORIGIN 1

#import "HNSTypedTensor.h"

@interface HNSTypedTensor (PrivateExtension)
- (unsigned) flatIndex:(NSArray *) index;	// calculate flat index position
@end

@implementation HNSTypedTensor (PrivateExtension)

- (unsigned) flatIndex:(NSArray *) index
{
	int i;
	unsigned ind=0;
	if([index count] != [rho count])
		[NSException raise:@"Index Error" format:@"Index has different dimension"]; // raise index dimension error
	for(i=0; i<[index count]; i++)
		{ // go through index and multiply each with next higher dimension
		int idx=[[index objectAtIndex:i] intValue]-origin;
		int dim=[[rho objectAtIndex:i] intValue];
		if(idx < 0 || idx >= dim)
			[NSException raise:@"Index Error" format:@"Index out of bounds"];
		ind=dim*ind+idx;	// merge into flat index
		}
	return ind;
}

@end

@implementation HNSTypedTensor

- (id) init
{ // any object type is permitted
	return [self initScalarForClass:[NSObject class]];
}

- (id) initScalarForClass:(Class) class;			// define permitted class (and subclasses)
{ // make scalar for that class
	self=[self initScalar:[NSNull null]];	// store single nil scalar value
	if(self)
		{
		[self setType:class];	// change permitted type
		}
	return self;
}

- (id) initScalar:(id) n;
{ // main initializer - scalar of given class
	self=[super init];
	if(self)
		{
		origin=INDEX_ORIGIN;
		rho=[[NSMutableArray arrayWithCapacity:0] retain];	// empty dimension vector
		data=[[NSMutableArray arrayWithCapacity:1] retain];	// data vector
		[self setType:[n class]];					// type of this object
		[self replaceObject:n];							// store object
		redim=YES;
		}
	return self;
}

- (id) initWithDimension:(NSArray *) dim forClass:(Class) class;	// array with class
{
	self=[self initScalarForClass:class];	// store single nil scalar value
	if(self)
		{
		[self setRho:dim];	// change dimension
		}
	return self;
}

- (id) initWithIndexGenerator:(int) n;
{ // 1...n (or 0..n-1)
	int i;
	if(n < 0)
		n=0;
	self=[self initWithDimension:[NSArray arrayWithObject:[NSNumber numberWithInt:n]] forClass:[NSNumber class]];
	if(self)
		{
		for(i=0; i<n; i++)
			[data addObject:[NSNumber numberWithInt:(i+origin)]];
		}
	return self;
}

- (id) iota:(int) n;
{ // 1...n (or 0..n-1)
	int i;
	if(n < 0)
		n=0;
	self=[self initWithDimension:[NSArray arrayWithObject:[NSNumber numberWithInt:n]] forClass:[NSNumber class]];
	if(self)
		{
		for(i=0; i<n; i++)
			[data addObject:[NSNumber numberWithInt:(i+origin)]];
		}
	return self;
}

- (void) dealloc
{
	[rho autorelease];
	[data autorelease];
	[super dealloc];
}

- (void) setType:(Class) c
{ // define for this class (and subclasses)
	if(c == nil)
		c=[NSObject class]; // raise exception or replace [NSObject class]
	type=c;
}

- (Class) getType
{
	return type;
}

- (NSArray *) rho
{
	return rho;
}

- (unsigned) rhorho
{
	return [rho count];
}

- (unsigned) count;
{ // get total size
	int i;
	unsigned size=1;
	for(i=0; i<[rho count]; i++)
		{ // go through index and determine total number of elements
		size*=[[rho objectAtIndex:i] intValue]-origin+1;	// dim index itself exists
		}
	return size;
}

- (unsigned) origin;
{
	return origin;
}

- (void) setOrigin:(unsigned) o;
{
	if(o > 1)
		[NSException raise:@"Value Error" format:@"Index origin may be 0 or 1 only"];
	origin=o;
}

- (void) setRho:(NSArray *) s
{ // redim
	int i;
	unsigned size=1;
	for(i=0; i<[s count]; i++)
		{ // go through index and determine total size
		unsigned d;
		d=[[s objectAtIndex:i] intValue]-origin+1;	// dim value itself also exists
		if((d>>24)*(size>>24) >= 50000)
			[NSException raise:@"Value Error" format:@"Tensor dimension overflow"];
		size*=d;
		}
	// reshape data
	[rho autorelease];
	rho=[s retain];
}

- (id) object;
{ // get scalar value
	if([rho count] > 0)
		[NSException raise:@"Value Error" format:@"Tensor is not a scalar"];
	return [data objectAtIndex:0];
}

- (NSArray *) array;
{ // get array value (includes scalars and empty tensors)
	if([rho count] > 1)
		[NSException raise:@"Value Error" format:@"Tensor is not a vector"];
	return data;	// in that case, we are simply an Array
}

- (id) objectAtIndex:(unsigned) i
{
	if([rho count] != 1)
		[NSException raise:@"Value Error" format:@"Tensor is not a vector"];
	// check index
	return [data objectAtIndex:i];
}

- (id) objectAtIndexList:(NSArray *) i
{
	unsigned int idx=[self flatIndex:i];
	id o;
	if(idx >= [data count])
		return nil;		// empty (i.e. not yet allocated)
	o=[data objectAtIndex:idx];
	if([o isKindOfClass:[NSNull class]])
		return nil;	// empty slot
	return o;
}

- (void) replaceObject:(id) val
{ // set scalar value
	[self replaceObjectAtIndexList:[NSArray array] withObject:val];	// empty array index -> scalar
}

- (void) replaceObjectAtIndexList:(NSArray *) i withObject:(id) obj;
{
	unsigned int idx=[self flatIndex:i];
	if(![obj isKindOfClass:type])
		[NSException raise:@"Value Error" format:@"Type mismatch"];
	if(idx >= [data count])
		{ // fill with NULL to required element
		id o=[NSNull null];
		unsigned ii=idx-[data count];	// create element for position up to and including idx
		[data addObject:o];				// make at least one entry
		while(ii-- > 0)
			[data addObject:o];			// remaining to fill up
		}
	[data replaceObjectAtIndex:idx withObject:obj];	// store value
}

- (HNSTypedTensor *) reduce:(SEL) selector;			// apply o=[o selector:x] for all elements x - has some built-in defaults for empty reduction dependent on selector
{
	return [self reduce:selector dim:0];	// first dimension
}

- (HNSTypedTensor *) reduce:(SEL) selector dim:(int) d;	// same along the d'th dimension
{
	int i;
	id r=nil;
	for(i=0; i<[[rho objectAtIndex:d] intValue]; i++)
		{
		if(i == 0)
			r=[data objectAtIndex:i];
		else
			r=[r performSelector:selector withObject:[data objectAtIndex:i]];	// reduce-operate
		}
	if(r == nil)
		{ // try to set default
		if(selector == @selector(CSAdd:))
			r=[NSNumber numberWithInt:0];
		if(selector == @selector(CSMul:))
			r=[NSNumber numberWithInt:1];
		}
	return r;
}

- (NSString *) description;
{
	return [NSString stringWithFormat:@"Tensor\norigin:\t%ld\nrhorho:\t%ld\nrho:\t%@\tcount:\t%ld", origin, [self rhorho], [self rho], [self count]];
}

@end